const nodemailer = require('nodemailer');
const { getDB } = require('../config/db');

// Configure Nodemailer
const transporter = nodemailer.createTransport({
    host: process.env.SMTP_HOST,
    port: process.env.SMTP_PORT,
    secure: process.env.SMTP_PORT == 465,
    auth: {
        user: process.env.SMTP_USER,
        pass: process.env.SMTP_PASS,
    },
});

// Verify connection
transporter.verify(function (error, success) {
    if (error) {
        console.error('SMTP Connection Error:', error);
    } else {
        console.log('SMTP Server is ready for broadcasting');
    }
});

/**
 * Sends a welcome email to a new subscriber
 */
const sendWelcomeEmail = async (email) => {
    const mailOptions = {
        from: process.env.SMTP_FROM,
        to: email,
        subject: 'Welcome to Nepal Pulse TV Newsletter!',
        html: `
            <div style="font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #eee; border-radius: 10px;">
                <h2 style="color: #ed1c24; text-align: center;">Nepal Pulse TV</h2>
                <p>Hello,</p>
                <p>Thank you for subscribing to our newsletter! You'll now receive the latest updates, breaking news, and trending stories from Nepal directly in your inbox.</p>
                <p>Stay informed with the most trusted source of news.</p>
                <div style="text-align: center; margin-top: 30px;">
                    <a href="${process.env.FRONTEND_URL || 'https://nepalpulsetv.com'}" style="background-color: #ed1c24; color: white; padding: 12px 25px; text-decoration: none; border-radius: 5px; font-weight: bold;">Visit Our Website</a>
                </div>
                <hr style="margin-top: 40px; border: 0; border-top: 1px solid #eee;" />
                <p style="font-size: 12px; color: #888; text-align: center;">
                    &copy; ${new Date().getFullYear()} Nepal Pulse TV. All rights reserved.<br/>
                    If you didn't subscribe, please ignore this email.
                </p>
            </div>
        `
    };

    return transporter.sendMail(mailOptions);
};

/**
 * Broadcasts a new article to all active subscribers
 */
const broadcastNewArticle = async (article) => {
    try {
        const db = getDB();
        const subscribers = await db.collection('subscribers').find({ status: 'active' }).toArray();

        if (!subscribers.length) return;

        const articleUrl = `${process.env.FRONTEND_URL || 'https://nepalpulsetv.com'}/article/${article.slug}`;
        const imageUrl = article.featuredImage || '';

        const mailOptions = {
            from: process.env.SMTP_FROM,
            subject: `New Story: ${article.title}`,
            html: `
                <div style="font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #eee; border-radius: 10px; color: #333;">
                    <div style="text-align: center; margin-bottom: 20px;">
                        <h1 style="color: #ed1c24; margin-bottom: 5px;">Nepal Pulse TV</h1>
                        <p style="font-size: 14px; color: #666; margin-top: 0;">Your Daily Pulse of News</p>
                    </div>
                    
                    ${imageUrl ? `<div style="margin-bottom: 20px;"><img src="${imageUrl}" alt="${article.title}" style="width: 100%; border-radius: 8px; max-height: 300px; object-fit: cover;"></div>` : ''}
                    
                    <h2 style="font-size: 22px; color: #111; margin-bottom: 15px;">${article.title}</h2>
                    
                    <p style="font-size: 16px; line-height: 1.6; color: #444; margin-bottom: 25px;">
                        ${article.excerpt || article.content.substring(0, 150).replace(/<[^>]*>/g, '') + '...'}
                    </p>
                    
                    <div style="text-align: center; margin-bottom: 30px;">
                        <a href="${articleUrl}" style="background-color: #ed1c24; color: white; padding: 15px 30px; text-decoration: none; border-radius: 5px; font-weight: bold; font-size: 16px;">Read Full Article</a>
                    </div>
                    
                    <div style="background-color: #f9f9f9; padding: 15px; border-radius: 8px; font-size: 13px; color: #666;">
                        <strong>Category:</strong> ${article.category}<br/>
                        <strong>Posted by:</strong> ${article.author || 'Nepal Pulse TV Team'}
                    </div>
                    
                    <hr style="margin: 30px 0; border: 0; border-top: 1px solid #eee;" />
                    
                    <p style="font-size: 11px; color: #999; text-align: center;">
                        You received this because you're subscribed to Nepal Pulse TV. <br/>
                        &copy; ${new Date().getFullYear()} Nepal Pulse TV. All rights reserved.
                    </p>
                </div>
            `
        };

        // Send to all subscribers (BCC for privacy and efficiency)
        const recipientEmails = subscribers.map(s => s.email);

        // Split recipients into batches of 50 to avoid SMTP limits
        const batchSize = 50;
        for (let i = 0; i < recipientEmails.length; i += batchSize) {
            const batch = recipientEmails.slice(i, i + batchSize);
            await transporter.sendMail({
                ...mailOptions,
                to: process.env.SMTP_USER, // Send to self
                bcc: batch // Recipients in BCC
            });
        }

        console.log(`Broadcasted article to ${recipientEmails.length} subscribers`);
    } catch (error) {
        console.error('Newsletter Broadcast Error:', error);
    }
};

module.exports = {
    sendWelcomeEmail,
    broadcastNewArticle
};
