const express = require('express');
const cors = require('cors');
const dotenv = require('dotenv');
const cookieParser = require('cookie-parser');
const path = require('path');
const { connectToMySQL } = require('./config/db');

dotenv.config({ path: path.join(__dirname, '.env') });

const app = express();
const PORT = process.env.PORT || 5000;


const allowedOrigins = [process.env.FRONTEND_URL, 'http://localhost:5173', 'http://localhost:5174'];
app.use(cors({
  origin: function (origin, callback) {

    if (!origin) return callback(null, true);
    if (allowedOrigins.indexOf(origin) === -1) {
      const msg = 'The CORS policy for this site does not allow access from the specified Origin.';
      return callback(new Error(msg), false);
    }
    return callback(null, true);
  },
  credentials: true
}));
app.use(express.json());
app.use(cookieParser());


const authRoutes = require('./routes/auth');
const breakingNewsRoutes = require('./routes/breakingNews');

app.use('/api/auth', authRoutes);
app.use('/api/breaking-news', breakingNewsRoutes);
app.use('/api/site-settings', require('./routes/siteSettings'));
app.use('/api/company-pages', require('./routes/companyPages'));
app.use('/api/upload', require('./routes/upload'));
app.use('/api/news', require('./routes/news'));
app.use('/api/categories', require('./routes/categories'));
app.use('/api/gold-silver', require('./routes/goldSilver'));
app.use('/api/careers', require('./routes/careers'));
app.use('/api/contact', require('./routes/contact'));
app.use('/api/newsletter', require('./routes/newsletter'));


app.use('/uploads', express.static(path.join(__dirname, 'uploads')));


app.get('/', (req, res) => {
  res.send('NepalPulseTV Backend is running');
});


async function startServer() {
  try {

    await connectToMySQL();

    app.listen(PORT, () => {
      console.log(`🚀 Server running on port ${PORT}`);
      console.log(`🔑 Super Admin URL: ${process.env.FRONTEND_URL}/super-admin`);
    });
  } catch (err) {
    console.error('Failed to start server:', err);
    process.exit(1);
  }
}

startServer();
