const express = require('express');
const router = express.Router();
const multer = require('multer');
const path = require('path');
const fs = require('fs');
const { authMiddleware, authorize } = require('../middleware/authMiddleware');


const CATEGORIES = ['team', 'news', 'category', 'ads', 'general'];


const storage = multer.diskStorage({
    destination: function (req, file, cb) {
        const category = req.query.category || 'general';

        
        const folderName = CATEGORIES.includes(category) ? category : 'general';
        const dir = path.join(__dirname, '../uploads', folderName);

        
        if (!fs.existsSync(dir)) {
            fs.mkdirSync(dir, { recursive: true });
        }

        cb(null, dir);
    },
    filename: function (req, file, cb) {
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        cb(null, uniqueSuffix + path.extname(file.originalname));
    }
});

const upload = multer({
    storage: storage,
    limits: { fileSize: 5 * 1024 * 1024 }, 
    fileFilter: (req, file, cb) => {
        const filetypes = /jpeg|jpg|png|webp/;
        const mimetype = filetypes.test(file.mimetype);
        const extname = filetypes.test(path.extname(file.originalname).toLowerCase());

        if (mimetype && extname) {
            return cb(null, true);
        }
        cb(new Error('Only images are allowed (jpeg, jpg, png, webp)'));
    }
});



router.post('/', authMiddleware, authorize('super-admin'), upload.single('image'), (req, res) => {
    try {
        if (!req.file) {
            return res.status(400).json({ success: false, message: 'No file uploaded' });
        }

        const category = req.query.category || 'general';
        const folderName = CATEGORIES.includes(category) ? category : 'general';

        
        const imageUrl = `/uploads/${folderName}/${req.file.filename}`;

        res.json({
            success: true,
            url: imageUrl,
            category: folderName,
            filename: req.file.filename,
            message: 'Image uploaded successfully'
        });
    } catch (error) {
        console.error('Upload error:', error);
        res.status(500).json({ success: false, message: 'Upload failed' });
    }
});

module.exports = router;
