const express = require('express');
const router = express.Router();
const { getDB } = require('../config/db');
const { v4: uuidv4 } = require('uuid');
const { authMiddleware } = require('../middleware/authMiddleware');


router.get('/', async (req, res) => {
    try {
        const db = getDB();
        const collection = db.collection('gold_silver_rates');
        const rates = await collection.find({}).sort({ date: -1 }).toArray();
        res.json(rates);
    } catch (err) {
        console.error('Error fetching gold/silver rates:', err);
        res.status(500).json({ message: 'Server error fetching rates' });
    }
});


router.get('/:id', async (req, res) => {
    try {
        const db = getDB();
        const collection = db.collection('gold_silver_rates');
        const rate = await collection.findOne({ id: req.params.id });
        if (!rate) {
            return res.status(404).json({ message: 'Rate not found' });
        }
        res.json(rate);
    } catch (err) {
        console.error('Error fetching rate:', err);
        res.status(500).json({ message: 'Server error fetching rate' });
    }
});


router.post('/', authMiddleware, async (req, res) => {
    try {
        const db = getDB();
        const collection = db.collection('gold_silver_rates');
        const { type, price, unit, date } = req.body;
        
        
        
        const dateRaw = date || new Date().toISOString();
        const rateDate = dateRaw.split('T')[0];

        
        
        
        console.log(`Checking for duplicate: Type=${type}, Date=${rateDate}`);

        
        const today = new Date();
        const todayStr = today.toLocaleDateString('en-CA'); 
        
        
        
        if (rateDate < todayStr) {
             return res.status(400).json({ 
                message: `Cannot post rates for past dates (${rateDate}). Please use today's date (${todayStr}) or a future date.` 
            });
        }

        const existingRate = await collection.findOne({ 
            type, 
            date: new RegExp(rateDate) 
        });
        
        if (existingRate) {
            console.log(`Duplicate found: ID=${existingRate.id}, Date=${existingRate.date}`);
            return res.status(400).json({ 
                message: `A ${type} rate for ${rateDate} already exists. Please edit the existing record instead.` 
            });
        }
        
        
        let change = '';
        try {
            
            
            const previousRate = await collection.find({ type })
                .sort({ date: -1, created_at: -1 })
                .limit(1)
                .toArray();
            
            if (previousRate.length > 0) {
                const currentPrice = parseFloat(price.replace(/,/g, ''));
                const oldPrice = parseFloat(previousRate[0].price.replace(/,/g, ''));
                
                if (!isNaN(currentPrice) && !isNaN(oldPrice)) {
                    const diff = currentPrice - oldPrice;
                    if (diff > 0) {
                        change = `+${diff}`;
                    } else if (diff < 0) {
                        change = `${diff}`;
                    } else {
                        change = '0';
                    }
                }
            }
        } catch (calcErr) {
            console.error('Error calculating change:', calcErr);
            
        }
        
        const newRate = {
            type,
            price,
            unit,
            change,
            date: rateDate,
            author: req.user.name,
            authorId: req.user.userId,
            created_at: new Date()
        };
        
        const result = await collection.insertOne(newRate);
        res.status(201).json({ ...newRate, id: result.insertedId });
    } catch (err) {
        console.error('Error creating rate:', err);
        res.status(500).json({ message: 'Server error creating rate' });
    }
});


router.put('/:id', authMiddleware, async (req, res) => {
    try {
        const db = getDB();
        const collection = db.collection('gold_silver_rates');
        const { id } = req.params;
        const { type, price, unit, change, date } = req.body;
        
        const existing = await collection.findOne({ id });
        if (!existing) return res.status(404).json({ message: 'Rate not found' });

        
        
        const newDateRaw = date || existing.date;
        const newDateKey = newDateRaw.split('T')[0];
        const newType = type || existing.type;

        
        
        if (date) {
            const today = new Date();
            const todayStr = today.toLocaleDateString('en-CA');
            if (newDateKey < todayStr) {
                return res.status(400).json({ 
                    message: `Cannot update rate to a past date (${newDateKey}). Please use today's date (${todayStr}) or a future date.` 
                });
            }
        }

        const conflict = await collection.findOne({ 
            type: newType, 
            date: new RegExp(newDateKey), 
            id: { $ne: id } 
        });

        if (conflict) {
            return res.status(400).json({ 
                message: `A ${newType} rate for ${newDateKey} already exists. Cannot update to this date.` 
            });
        }

        const updateData = {
            type,
            price,
            unit,
            change,
            date,
            updatedBy: req.user.name
        };

        await collection.findOneAndUpdate({ id }, { $set: updateData });
        
        res.json({ id, ...existing, ...updateData });
    } catch (err) {
        console.error('Error updating rate:', err);
        res.status(500).json({ message: 'Server error updating rate' });
    }
});


router.delete('/:id', authMiddleware, async (req, res) => {
    try {
        const db = getDB();
        const collection = db.collection('gold_silver_rates');
        const { id } = req.params;
        
        const result = await collection.deleteOne({ id });
        if (result.deletedCount === 0) {
            return res.status(404).json({ message: 'Rate not found' });
        }
        
        res.json({ message: 'Rate deleted successfully' });
    } catch (err) {
        console.error('Error deleting rate:', err);
        res.status(500).json({ message: 'Server error deleting rate' });
    }
});

module.exports = router;
