const express = require('express');
const router = express.Router();
const { authMiddleware, authorize } = require('../middleware/authMiddleware');
const { getDB } = require('../config/db');
const { v4: uuidv4 } = require('uuid');


router.post('/', async (req, res) => {
    try {
        const { name, email, subject, message } = req.body;

        if (!name || !email || !message) {
            return res.status(400).json({
                success: false,
                message: 'Name, email, and message are required'
            });
        }

        // Email validation regex
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(email)) {
            return res.status(400).json({
                success: false,
                message: 'Please provide a valid email address'
            });
        }

        const db = getDB();
        const newMessage = {
            id: uuidv4(),
            name,
            email,
            subject: subject || 'No Subject',
            message,
            status: 'unread',
            created_at: new Date()
        };




        const result = await db.collection('contact_messages').insertOne(newMessage);

        res.status(201).json({
            success: true,
            message: 'Message sent successfully',
            data: { id: result.insertedId, ...newMessage }
        });
    } catch (error) {
        console.error('Contact error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to send message',
            error: error.message
        });
    }
});


router.get('/', authMiddleware, authorize('admin', 'super-admin'), async (req, res) => {
    try {
        console.log('GET /api/contact request received');
        console.log('User:', req.user);

        const db = getDB();
        const page = parseInt(req.query.page) || 1;
        const limit = parseInt(req.query.limit) || 20;
        const skip = (page - 1) * limit;

        const messages = await db.collection('contact_messages')
            .find({})
            .sort({ created_at: -1 })
            .skip(skip)
            .limit(limit)
            .toArray();

        const total = await db.collection('contact_messages').countDocuments({});



        res.json({
            success: true,
            data: messages,
            pagination: {
                total,
                page,
                limit,
                totalPages: Math.ceil(total / limit)
            }
        });
    } catch (error) {
        console.error('Fetch messages error:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching messages',
            error: error.message
        });
    }
});


router.delete('/:id', authMiddleware, authorize('admin', 'super-admin'), async (req, res) => {
    try {
        const db = getDB();
        const result = await db.collection('contact_messages').deleteOne({ id: req.params.id });

        if (result.deletedCount === 0) {
            return res.status(404).json({ success: false, message: 'Message not found' });
        }

        res.json({ success: true, message: 'Message deleted' });
    } catch (error) {
        console.error('Delete message error:', error);
        res.status(500).json({ success: false, message: 'Error deleting message' });
    }
});

router.put('/:id/read', authMiddleware, authorize('admin', 'super-admin'), async (req, res) => {
    try {
        const db = getDB();
        const result = await db.collection('contact_messages').updateOne(
            { id: req.params.id },
            { $set: { status: 'read' } }
        );

        if (result.matchedCount === 0) {
            return res.status(404).json({ success: false, message: 'Message not found' });
        }

        res.json({ success: true, message: 'Message marked as read' });
    } catch (error) {
        console.error('Update message error:', error);
        res.status(500).json({ success: false, message: 'Error updating message' });
    }
});

module.exports = router;
