const express = require('express');
const router = express.Router();
const { authMiddleware } = require('../middleware/authMiddleware');
const { getDB } = require('../config/db');
const { v4: uuidv4 } = require('uuid');


router.get('/', async (req, res) => {
    try {
        const db = getDB();
        const careers = await db.collection('careers')
            .find({})
            .sort({ created_at: -1 })
            .toArray();

        
        const mappedCareers = careers.map(job => ({
            id: job._id.toString(),
            ...job
        }));

        res.json({
            success: true,
            data: mappedCareers
        });
    } catch (error) {
        res.status(500).json({
            success: false,
            message: 'Error fetching careers',
            error: error.message
        });
    }
});


router.get('/:id', async (req, res) => {
    try {
        const db = getDB();
        const job = await db.collection('careers').findOne({ _id: req.params.id });

        if (!job) return res.status(404).json({ success: false, message: 'Job not found' });

        const mappedJob = {
            id: job._id.toString(),
            ...job
        };

        res.json({ success: true, data: mappedJob });
    } catch (error) {
        res.status(500).json({ success: false, message: 'Error fetching job', error: error.message });
    }
});


router.post('/', authMiddleware, async (req, res) => {
    try {
        const {
            title,
            department,
            location,
            type,
            description,
            requirements,
            deadline,
            status = 'active',
            company_name = 'NepalPulseTV',
            company_logo = '',
            application_link = '',
            is_external = false
        } = req.body;

        if (!title || !description) {
            return res.status(400).json({
                success: false,
                message: 'Title and description are required'
            });
        }

        const db = getDB();
        const newJob = {
            title,
            department,
            location,
            type,
            description,
            requirements, 
            deadline,
            status,
            company_name,
            company_logo,
            application_link,
            is_external,
            posted_at: new Date().toISOString()
        };

        const result = await db.collection('careers').insertOne(newJob);

        res.status(201).json({
            success: true,
            message: 'Job created successfully',
            data: {
                id: result.id,
                ...newJob
            }
        });
    } catch (error) {
        res.status(500).json({
            success: false,
            message: 'Error creating job',
            error: error.message
        });
    }
});


router.put('/:id', authMiddleware, async (req, res) => {
    try {
        const {
            title,
            department,
            location,
            type,
            description,
            requirements,
            deadline,
            status,
            company_name,
            company_logo,
            application_link,
            is_external
        } = req.body;

        const db = getDB();
        
        
        const updateData = {};
        if (title) updateData.title = title;
        if (department) updateData.department = department;
        if (location) updateData.location = location;
        if (type) updateData.type = type;
        if (description) updateData.description = description;
        if (requirements) updateData.requirements = requirements;
        if (deadline) updateData.deadline = deadline;
        if (status) updateData.status = status;
        if (company_name) updateData.company_name = company_name;
        if (company_logo !== undefined) updateData.company_logo = company_logo;
        if (application_link !== undefined) updateData.application_link = application_link;
        if (is_external !== undefined) updateData.is_external = is_external;
        
        updateData.updated_at = new Date().toISOString();

        const result = await db.collection('careers').updateOne(
            { _id: req.params.id },
            { $set: updateData }
        );

        if (result.matchedCount === 0) {
            return res.status(404).json({ success: false, message: 'Job not found' });
        }

        res.json({
            success: true,
            message: 'Job updated successfully'
        });
    } catch (error) {
        res.status(500).json({
            success: false,
            message: 'Error updating job',
            error: error.message
        });
    }
});


router.delete('/:id', authMiddleware, async (req, res) => {
    try {
        const db = getDB();
        const result = await db.collection('careers').deleteOne({ _id: req.params.id });

        if (result.deletedCount === 0) {
            return res.status(404).json({ success: false, message: 'Job not found' });
        }

        res.json({
            success: true,
            message: 'Job deleted successfully'
        });
    } catch (error) {
        res.status(500).json({
            success: false,
            message: 'Error deleting job',
            error: error.message
        });
    }
});

module.exports = router;
