const express = require('express');
const router = express.Router();
const { authMiddleware, authorize } = require('../middleware/authMiddleware');
const { getDB } = require('../config/db');


router.get('/', async (req, res) => {
    try {
        const db = getDB();
        
        let news = [];
        try {
            news = await db.collection('breaking_news')
                .find({})
                .sort({ created_at: -1 })
                .toArray();
        } catch (dbError) {
            console.warn('Error fetching from DB, returning empty:', dbError.message);
        }

        res.json({
            success: true,
            data: news.map(n => ({
                id: n.id,
                text: n.text || n.data?.text || n.title || n.data?.title || '', 
                createdAt: n.created_at
            }))
        });
    } catch (error) {
        res.status(500).json({
            success: false,
            message: 'Error fetching breaking news',
            error: error.message
        });
    }
});


router.post('/', authMiddleware, authorize('super-admin'), async (req, res) => {
    try {
        const { text } = req.body;

        if (!text) {
            return res.status(400).json({
                success: false,
                message: 'Breaking news text is required'
            });
        }

        const db = getDB();
        const newNews = { text };

        const result = await db.collection('breaking_news').insertOne(newNews);

        res.status(201).json({
            success: true,
            message: 'Breaking news created successfully',
            data: {
                id: result.insertedId,
                text,
                createdAt: new Date()
            }
        });
    } catch (error) {
        res.status(500).json({
            success: false,
            message: 'Error creating breaking news',
            error: error.message
        });
    }
});


router.put('/:id', authMiddleware, authorize('super-admin'), async (req, res) => {
    try {
        const { id } = req.params;
        const { text } = req.body;

        const db = getDB();

        
        const news = await db.collection('breaking_news').findOne({ id });
        if (!news) {
            return res.status(404).json({
                success: false,
                message: 'Breaking news not found'
            });
        }

        await db.collection('breaking_news').updateOne(
            { id },
            { $set: { text } }
        );

        res.json({
            success: true,
            message: 'Breaking news updated successfully',
            data: { ...news, text }
        });
    } catch (error) {
        res.status(500).json({
            success: false,
            message: 'Error updating breaking news',
            error: error.message
        });
    }
});


router.delete('/:id', authMiddleware, authorize('super-admin'), async (req, res) => {
    try {
        const { id } = req.params;
        const db = getDB();

        const result = await db.collection('breaking_news').deleteOne({ id });

        if (result.deletedCount === 0) {
            return res.status(404).json({
                success: false,
                message: 'Breaking news not found'
            });
        }

        res.json({
            success: true,
            message: 'Breaking news deleted successfully'
        });
    } catch (error) {
        res.status(500).json({
            success: false,
            message: 'Error deleting breaking news',
            error: error.message
        });
    }
});

module.exports = router;
